/* $Id: spans.c,v 1.23 1998/09/16 02:05:18 ericb Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Thoroughly test span changes */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"
#include "err1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	1024

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: spans.c,v 1.23 1998/09/16 02:05:18 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int data_mode, int fft)
{
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_data_mode(hw, group, data_mode));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    if (fft)
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, double clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 1)
	(void) printf("Block available found\n");

    return 0;
}

static int
get_scan(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	 double clock_freq, int fft, int verbose, double span)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    int     type, chan;

    /* Wait for block available */
    CHECK(wait_block_avail(hw, group, verbose, BLOCKSIZE, span));

    /* Read the data */
    for (type = 0; type < (fft ? 2 : 1); type++)
	for (chan = 0; chan < nchan; chan++)
	{
	    if (verbose > 3)
		(void) printf("Reading chan %d type %d\n", chan, type);

	    CHECK(e1432_read_float64_data(hw, chan_list[chan],
					  type == 0 ? E1432_TIME_DATA :
					  E1432_FREQ_DATA, buffer,
					  BLOCKSIZE, &trailer,
					  &count));
	    if (count != BLOCKSIZE)
	    {
		(void) fprintf(stderr, "%s: e1432_read_float64_data: "
			       "actual count was %ld, expected %d\n",
			       progname, count, BLOCKSIZE);
		return -1;
	    }

	    CHECK(check_trailer(&trailer, clock_freq, span,
				chan_list[chan], type));
	}

    return 0;
}

#ifdef	__lint
#define	SPANCHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	(void) fprintf(stderr,\
		       "   Sent span %11.4f, expected  %11.4f\n"\
		       "   Clock freq %g, running %d, fft %d\n",\
		       send, expected, clock_freq, running, fft);\
	return _s;\
    }\
} while (func)
#else
#define	SPANCHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	(void) fprintf(stderr,\
		       "   Sent span %11.4f, expected  %11.4f\n"\
		       "   Clock freq %g, running %d, fft %d\n",\
		       send, expected, clock_freq, running, fft);\
	return _s;\
    }\
} while (0)
#endif

static int
do_span(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	double clock_freq, int fft, int verbose, int running,
	double send, double expected)
{
    FLOATSIZ32 actual;
    double ratio;

    if (verbose > 1)
    {
	(void) printf("Sending span %11.4f, expecting %11.4f,",
		      send, expected);
	(void) fflush(stdout);
    }
    SPANCHECK(e1432_set_span(hw, group, send));
    SPANCHECK(e1432_get_span(hw, group, &actual));
    if (verbose > 1)
	(void) printf(" actual %11.4f\n", actual);
    ratio = actual / expected;
    if (ratio > 1.0001 || ratio < 0.9999)
    {
	(void) fprintf(stderr,
		       "%s: span mismatch\n"
		       "   Sent span %11.4f, expected  %11.4f, actual %11.4f\n"
		       "   Clock freq %g, running %d, fft %d\n",
		       progname, send, expected, actual,
		       clock_freq, running, fft);
	return -1;
    }

    if (running)
	SPANCHECK(get_scan(hw, group, nchan, chan_list,
			   clock_freq, fft, verbose, expected));

    return 0;
}

static int
do_bad_span(E1432ID hw, int group, double clock_freq, int fft,
	    int verbose, int running, double span)
{
    int     status;

    if (verbose > 1)
	(void) printf("Sending span %11.4f, expecting error\n", span);

    CHECK(e1432_print_errors(0));
    status = e1432_set_span(hw, group, span);
    CHECK(e1432_print_errors(1));
    if (status != ERR1432_ILLEGAL_SPAN)
    {
	(void) fprintf(stderr,
		       "%s: illegal span returned status %d\n",
		       progname, status);
	(void) fprintf(stderr,
		       "   Sent span %11.4f\n"
		       "   Clock freq %g, running %d, fft %d\n",
		       span, clock_freq, running, fft);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    double clock_freq, int fft, int verbose)
{
    double  top_span, span;
    int     i, avoid_top_span;

    CHECK(e1432_set_clock_freq(hw, group, clock_freq));
    top_span = clock_freq / 2.56;
    avoid_top_span = clock_freq > 51201;

    if (verbose)
	(void) printf("Clock freq %g, no measurement running\n",
		      clock_freq);
    CHECK(e1432_reset_measure(hw, group));
    span = top_span;
    CHECK(do_bad_span(hw, group, clock_freq,
		      fft, verbose, 0, top_span * 1.001));
    for (i = 0; i < 9; i++, span /= 2)
    {
	if (i == 0 && avoid_top_span)
	    continue;

	CHECK(do_span(hw, group, nchan, chan_list,
		      clock_freq, fft, verbose, 0, span, span));
	CHECK(do_span(hw, group, nchan, chan_list,
		      clock_freq, fft, verbose, 0, span * 0.9, span));
	CHECK(do_span(hw, group, nchan, chan_list,
		      clock_freq, fft, verbose, 0, span / 5,
		      span / 5));
	CHECK(do_span(hw, group, nchan, chan_list,
		      clock_freq, fft, verbose, 0, span / 5 * 0.9,
		      span / 5));

	if (i > 3 || (i == 3 && !avoid_top_span))
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 0, span * 1.001,
			  span * 8 / 5));
	else if (i == 2 || (i == 1 && !avoid_top_span))
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 0, span * 1.001,
			  span * 2));

	if (i < 7)
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 0,
			  span / 5 * 1.001, span / 4));
    }

    /* Start a measurement with a non-divide-by-five span */
    if (verbose)
	(void) printf("Clock freq %g, non-divide-by-five spans on the fly\n",
		      clock_freq);
    CHECK(do_span(hw, group, nchan, chan_list, clock_freq,
		  fft, verbose, 0, top_span / 2, top_span / 2));
    CHECK(e1432_init_measure(hw, group));

    span = top_span;
    for (i = 0; i < 9; i++, span /= 2)
    {
	if (i == 0 && avoid_top_span)
	    continue;

	CHECK(do_span(hw, group, nchan, chan_list,
		      clock_freq, fft, verbose, 1, span, span));
	CHECK(do_span(hw, group, nchan, chan_list,
		      clock_freq, fft, verbose, 1, span * 0.9, span));

	if (i > 1 || (i == 1 && !avoid_top_span))
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 1, span * 1.001,
			  span * 2));

	if (i < 7)
	{
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 1, span / 5,
			  span / 4));
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 1, span / 5 * 0.9,
			  span / 4));
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 1,
			  span / 5 * 1.001, span / 4));
	}
    }

    /* Start a measurement with a divide-by-five span */
    if (verbose)
	(void) printf("Clock freq %g, divide-by-five spans on the fly\n",
		      clock_freq);
    CHECK(e1432_reset_measure(hw, group));
    CHECK(do_span(hw, group, nchan, chan_list, clock_freq,
		  fft, verbose, 0, top_span / 10, top_span / 10));
    CHECK(e1432_init_measure(hw, group));

    span = top_span;
    CHECK(do_bad_span(hw, group, clock_freq,
		      fft, verbose, 1, top_span));
    for (i = 0; i < 9; i++, span /= 2)
    {
	if (i == 0 && avoid_top_span)
	    continue;

	if (i > 3 || (i == 3 && !avoid_top_span))
	{
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 1, span,
			  span * 8 / 5));
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 1, span * 0.9,
			  span * 8 / 5));
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 1, span * 1.001,
			  span * 8 / 5));
	}
	else if (i <= 2)
	{
	    CHECK(do_bad_span(hw, group, clock_freq,
			      fft, verbose, 1, span));
	    CHECK(do_bad_span(hw, group, clock_freq,
			      fft, verbose, 1, span * 0.9));
	    CHECK(do_bad_span(hw, group, clock_freq,
			      fft, verbose, 1, span * 1.001));
	}

	CHECK(do_span(hw, group, nchan, chan_list,
		      clock_freq, fft, verbose, 1, span / 5,
		      span / 5));
	CHECK(do_span(hw, group, nchan, chan_list,
		      clock_freq, fft, verbose, 1, span / 5 * 0.9,
		      span / 5));

	if (i < 7 && i > 0 && !(i == 1 && avoid_top_span))
	    CHECK(do_span(hw, group, nchan, chan_list,
			  clock_freq, fft, verbose, 1,
			  span / 5 * 1.001, span * 2 / 5));
	if (i == 0)
	    CHECK(do_bad_span(hw, group, clock_freq,
			      fft, verbose, 1, span / 5 * 1.001));
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-bcfuvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t-b: Use block mode\n"
		   "\t-c: Use continuous mode\n"
		   "\t-f: Set up for fft data\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    char   *p;
    int     c, i, data_mode, fft, nmod, verbose;
    int     group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
    fft = 0;
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "bcfL:n:N:uvV")) != -1)
	switch (c)
	{
	case 'b':
	    data_mode = E1432_BLOCK_MODE;
	    break;
	case 'c':
	    data_mode = E1432_CONTINUOUS_MODE;
	    break;
	case 'f':
	    fft = 1;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, data_mode, fft) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, 65536.0, fft, verbose) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, 51200.0, fft, verbose) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, 40960.0, fft, verbose) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, 96000.0, fft, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
